% Matlab program for Diamond's OLG model.
% Matlab program to draw phase diagrams in the baseline Diamond overlapping generations model.
% By Marc P. B. Klemp, November 2009.

global Dist_util Util_type Prod_type grid;
global a b d h m rho n p Dd De Df Dg;

% PROGRAM OPTIONS

% The following combinations of parameters generate a benchmark case
% Case 1: Util_type = 1, Prod_type = 2, a = 10, b = 0.3, p = unused, d = 0.02, m = 1, rho = 0.01, n = 0.01 and xmax=ymax=10 for viewing
%
% The following combinations of parameters generate poverty-traps
% Case 1: Util_type = 1, Prod_type = 1, a = 15, b = 0.3, p = -4, d = 0.02, m = 1, rho = 2, n = 0.01 and xmax=ymax=5 for viewing
%
% The following combinations of parameters generate multiple intertemporal equilibria:
% Case 1: Util_type = 1, Prod_type = 1, a = 18, b = 0.3, p = -4, d = 0.02, m = 5, rho = 2, n = 0.01 and xmax=ymax=10 for viewing
% Case 2: Util_type = 3, everything else as in case 1 

Dist_util = 0; % Set to 1 to use distinct period utility functions for the two periods.

Util_type = 1; % Choose the type of instantaneous utility function here
               % 1: CRRA:                                           log(c) when m=1 and (c.^(1-m))/(1-m) when m!=1 and m>0
               % 2: CRRA with subsistence consumption:              log(c-h) for all h when m = 1 and (c.^(1-m))/(1-m) for h>=0 when m!=1 and m>0
               % 3: CARA:                                           -exp(-m*c) for m>0
               % 4: CARA-like, satisfying the "no fast" assumption: -1/c when c <  0.70... and -exp(-m*c) for m>0 otherwise
               
Prod_type = 1; % Choose the intensive production function here
               % 1: CES:          a*(b*k^p+(1-b))^(1/p)
               % 2: Cobb-Douglas: a*k^b
               % 3: Unnamed:      b+a*k/(1+k)

Line_f    = 1; % Set to 1 to plot f(k_t)/(1+n). This line is dashed.

Line_w    = 1; % Set to 1 to plot w(k_t)/(1+n). This line is dash/dot'ed.

Line_45_d = 1; % Set to 1 to draw the 45-degree line on the phase diagram.

Show_para = 1; % Set to 1 to show the parameter values on the reaction curve.
               
grid = 200;    % Resolution of the plots, the program calculates grid^2 mesh points
xmin = 0;      % The minimal value of k_t to be used in the plots (in case xmin=0 result in problems, use something like xmin=0.01)
ymin = 0;      % The minimal value of k_tplus1 to be used in the plots (in case ymin=0 result in problems, use something like ymin=0.01)
xmax = 5;      % The maximal value of k_t to be used on the plots
ymax = 7;      % The maxmial value of k_tplus1 to be used on the plots

% MODEL PARAMETERS

a   = 7;       % Parameter of all three production functions
b   = 0.33;    % Parameter of all three production functions
p   = -4.0;    % Parameter of the CES production function
d   = 0.6;     % Capital depreciation rate
h   = 0;       % Subsistence consumption in the subsistence CRRA utility function
m   = 8;       % Parameter of all three utility functions
rho = 0.4;     % Utility discount rate
n   = 0.2;     % Population growth rate

Dd = 1;
De = 1;
Df = 1;
Dg = 1;

% SETUP GRID

[x,y] = meshgrid(xmin:(xmax-xmin)/grid:xmax,ymin:(ymax-ymin)/grid:ymax);     % Generate a grid from xmin to xmax on the horizontal axis and from ymin
                                                                             % to ymax on the vertical axis with grid number of points on each axis

% CALCULATE SURFACE

clear z;       % Clear z in case grid has been changed between executions

for i = 1:grid+1                                                             % For each point on the vertical axis...
    for j = 1:grid+1                                                         % ...given each point on the horizontal axis...
        z(j,i) = G(xmin+(i-1)*(xmax-xmin)/grid,ymin+(j-1)*(ymax-ymin)/grid); % ...calculate the value of the G-function corresponding to the value of
    end                                                                      % the corresponding x and y values
end

% PLOT FIGURES

figure(1), mesh(x,y,z);               % Plot the surface generated by the mesh combined with the corresponding values of the G function
           title('Plot of G(k_t, k_{t+1})');
           xlabel('k_t');
           ylabel('k_{t+1}');

figure(2), contour(x,y,z,[0,0], 'k'); % Plot the relevant countour only, where G(k_t,k_tplus1)=0 (i.e., essentially, plot G(k_t,k_tplus1)=0 implicitly)
           title('Reaction curve');
           xlabel('k_t');
           ylabel('k_{t+1}');
if Line_f == 1
           hold on;
           yf = f(0:(xmax-xmin)/grid:xmax) / (1+n);
           plot(0:(xmax-xmin)/grid:xmax,yf,'--k');
           hold off;
end
if Line_w == 1
           hold on;
           yw = W(0:(xmax-xmin)/grid:xmax) / (1+n);
           plot(0:(xmax-xmin)/grid:xmax,yw,'-.k');
           hold off;
end
if Line_45_d == 1
           hold on;
           X = [0, max([xmax ymax])];
           Y = [0, max([xmax ymax])];
           plot(X, Y, ':k');
           hold off;
end

if Show_para == 1
    text(xmin+(xmax-xmin)*0.25,ymin+(ymax-ymin)*0.10,['Dist\_util=', num2str(Dist_util), ', Util\_type=', num2str(Util_type), ', Prod\_type=', num2str(Prod_type)'])
    text(xmin+(xmax-xmin)*0.25,ymin+(ymax-ymin)*0.05,['a=', num2str(a), ', b=', num2str(b) ', p=', num2str(p) ', d=', num2str(d) ', h=', num2str(h) ', m=', num2str(m) ', rho=', num2str(rho) ', n=', num2str(n)])
end

run pp % Print parameters
run pw % Print warnings